package com.crankuptheamps.examples;

import java.lang.Thread;
import java.util.Map;
import com.crankuptheamps.client.*;
import com.crankuptheamps.client.exception.*;

/**
 * FIXShredderSubscriber
 *
 * This sample subscribes to FIX messages. The program flow is simple:
 *
 * * Connect to AMPS
 * * Logon
 * * Connect to the "messages" topic and shred
 * * messages to a key value data structure.
 * * Output all messages received to the console
 *
 * This sample doesn't include error handling or connection
 * retry logic.
 *
 * (c) 2013-2015 60East Technologies, Inc. All rights reserved.
 * This file is a part of the AMPS Evaluation Kit.
 */
public class FIXShredderSubscriber {

  // The location of the AMPS server.
  private static final String uri_ = "tcp://127.0.0.1:9007/amps/fix";

  /**
   * main method.
   * 
   * @param args -- No command line options read.
   */
   public static void main(String[] args)
   {
        System.out.println("Starting the console subscriber.");

        try (HAClient client = new HAClient("ConsoleSubscriber");) {

          DefaultServerChooser sc = new DefaultServerChooser();
          sc.add(uri_);
          client.setServerChooser(sc);
          client.connectAndLogon();
          System.out.println("connected..");

              // subscribe to the test-topic topic.
              // when a message arrives, print the message.
              try (MessageStream ms = client.subscribe("messages");) {
                // Create a shredder -- since this just returns
                // the Map, we can reuse the same shredder.
                FIXShredder shredder = new FIXShredder((byte)1);

                for (Message m : ms)
                {
                   // Skip messages with no data.
                   if (m.getCommand() != Message.Command.SOW &&
                       m.getCommand() != Message.Command.Publish) continue;

                   System.out.println("Got a message");

                   // Shred the message into a Map
                   Map<Integer,CharSequence> fields = shredder.toMap(m.getData());
                   // Iterate over the keys in the map and print the key and data
                   for (Integer key : fields.keySet())
                   {
                       System.out.println("  " +key + "=" + fields.get(key));
                   }
                }
              }
       }
       catch (AMPSException e)
       {
          System.err.println(e.getLocalizedMessage());
          e.printStackTrace(System.err);
       }
  }
}
