package com.crankuptheamps.examples;

import com.crankuptheamps.client.*;
import com.crankuptheamps.client.exception.*;
import java.text.MessageFormat;


/**
 * SOWConsolePublisher
 *
 * This sample publishes messages to a topic in AMPS that
 * maintains a state-of-the-world (SOW) database.
 * The program flow is simple:
 *
 * * Connect to AMPS
 * * Logon
 * * Publish 100 messages to the "messages-sow" topic
 * * Publish another message with a duplicate messageNumber to the topic,
 * effectively updating that message.
 *
 * The "messages-sow" topic is configured in config/sample.xml to
 * maintain a SOW database, where each messageNumber is a unique
 * message.
 *
 * This sample doesn't include error handling or connection
 * retry logic.
 *
 * (c) 2013 60East Technologies, Inc. All rights reserved.
 */
public class SOWConsolePublisher {

  // The location of the AMPS server.
  private static final String uri_ = "tcp://127.0.0.1:9007/amps/json";

  /**
   * main method.
   *
   * @param args -- No command line options read.
   */
  public static void main(String[] args) {

    String dataFormat = "{\"messageNumber\" : %d" +
                        ", \"message\" : \"%s\"}"; 

    try (HAClient client = new HAClient("SOWConsolePublisher");) {

      // Register a failed write handler to be notified of publish errors
      // NOTE: must also register a publish store to receive a copy of
      // the message that failed.
      client.setFailedWriteHandler(new FailedWriteHandler() {

        @Override
        public void failedWrite(Message message, int reason) {
          System.out.println("FAILED PUBLISH: reason = " + reason + ", message = " + message);
        }

      });

      // Create a custom server chooser
      DefaultServerChooser serverChooser = new DefaultServerChooser();
      serverChooser.add(uri_);

      // Set the server chooser for the HAClient
      client.setServerChooser(serverChooser);
      // connect to the AMPS server and logon
      client.connectAndLogon();
      System.out.println("SOW Publisher Connected");

      // send 100 messages with unique message numbers
      // to fill the SOW database

      for (int number = 0 ; number< 100; ++number)
      {
        client.publish("messages-sow",
            String.format(dataFormat, number, "Hello, World!"));
      }
      // Now make a change to message 5

      client.publish("messages-sow",
          String.format(dataFormat, 5, "This is new information."));
      // Wait for all published messages acknowledged as persisted on the server
      client.publishFlush(10000L);
    }
    catch (AMPSException e)
    {
      System.err.println(e.getLocalizedMessage());
      e.printStackTrace(System.err);
    }

  }

}
